// Copyright 1998-2015 Epic Games, Inc. All Rights Reserved.

#include "Cookbook_Chapter9.h"
#include "Cookbook_Chapter9Character.h"

#include "MySaveGame.h"
#include "Kismet/GameplayStatics.h"
#include "Engine.h" //GEngine

//////////////////////////////////////////////////////////////////////////
// ACookbook_Chapter9Character

ACookbook_Chapter9Character::ACookbook_Chapter9Character()
{
	PrimaryActorTick.bCanEverTick = true; 

	// Set size for collision capsule
	GetCapsuleComponent()->InitCapsuleSize(42.f, 96.0f);

	// set our turn rates for input
	BaseTurnRate = 45.f;
	BaseLookUpRate = 45.f;

	// Don't rotate when the controller rotates. Let that just affect the camera.
	bUseControllerRotationPitch = false;
	bUseControllerRotationYaw = false;
	bUseControllerRotationRoll = false;

	// Configure character movement
	GetCharacterMovement()->bOrientRotationToMovement = true; // Character moves in the direction of input...	
	GetCharacterMovement()->RotationRate = FRotator(0.0f, 540.0f, 0.0f); // ...at this rotation rate
	GetCharacterMovement()->JumpZVelocity = 600.f;
	GetCharacterMovement()->AirControl = 0.2f;

	// Create a camera boom (pulls in towards the player if there is a collision)
	CameraBoom = CreateDefaultSubobject<USpringArmComponent>(TEXT("CameraBoom"));
	CameraBoom->AttachTo(RootComponent);
	CameraBoom->TargetArmLength = 300.0f; // The camera follows at this distance behind the character	
	CameraBoom->bUsePawnControlRotation = true; // Rotate the arm based on the controller

	// Create a follow camera
	FollowCamera = CreateDefaultSubobject<UCameraComponent>(TEXT("FollowCamera"));
	FollowCamera->AttachTo(CameraBoom, USpringArmComponent::SocketName); // Attach the camera to the end of the boom and let the boom adjust to match the controller orientation
	FollowCamera->bUsePawnControlRotation = false; // Camera does not rotate relative to arm

	// Note: The skeletal mesh and anim blueprint references on the Mesh component (inherited from Character) 
	// are set in the derived blueprint asset named MyCharacter (to avoid direct content references in C++)
}

//////////////////////////////////////////////////////////////////////////
// Input

void ACookbook_Chapter9Character::SetupPlayerInputComponent(class UInputComponent* InputComponent)
{
	// Set up gameplay key bindings
	check(InputComponent);
	InputComponent->BindAction("Jump", IE_Pressed, this, &ACharacter::Jump);
	InputComponent->BindAction("Jump", IE_Released, this, &ACharacter::StopJumping);

	InputComponent->BindAxis("MoveForward", this, &ACookbook_Chapter9Character::MoveForward);
	InputComponent->BindAxis("MoveRight", this, &ACookbook_Chapter9Character::MoveRight);

	// We have 2 versions of the rotation bindings to handle different kinds of devices differently
	// "turn" handles devices that provide an absolute delta, such as a mouse.
	// "turnrate" is for devices that we choose to treat as a rate of change, such as an analog joystick
	InputComponent->BindAxis("Turn", this, &APawn::AddControllerYawInput);
	InputComponent->BindAxis("TurnRate", this, &ACookbook_Chapter9Character::TurnAtRate);
	InputComponent->BindAxis("LookUp", this, &APawn::AddControllerPitchInput);
	InputComponent->BindAxis("LookUpRate", this, &ACookbook_Chapter9Character::LookUpAtRate);

	// handle touch devices
	InputComponent->BindTouch(IE_Pressed, this, &ACookbook_Chapter9Character::TouchStarted);
	InputComponent->BindTouch(IE_Released, this, &ACookbook_Chapter9Character::TouchStopped);
}


void ACookbook_Chapter9Character::TouchStarted(ETouchIndex::Type FingerIndex, FVector Location)
{
	// jump, but only on the first touch
	if (FingerIndex == ETouchIndex::Touch1)
	{
		Jump();
	}
}

void ACookbook_Chapter9Character::TouchStopped(ETouchIndex::Type FingerIndex, FVector Location)
{
	if (FingerIndex == ETouchIndex::Touch1)
	{
		StopJumping();
	}
}

void ACookbook_Chapter9Character::TurnAtRate(float Rate)
{
	// calculate delta for this frame from the rate information
	AddControllerYawInput(Rate * BaseTurnRate * GetWorld()->GetDeltaSeconds());
}

void ACookbook_Chapter9Character::LookUpAtRate(float Rate)
{
	// calculate delta for this frame from the rate information
	AddControllerPitchInput(Rate * BaseLookUpRate * GetWorld()->GetDeltaSeconds());
}

void ACookbook_Chapter9Character::MoveForward(float Value)
{
	if ((Controller != NULL) && (Value != 0.0f))
	{
		// find out which way is forward
		const FRotator Rotation = Controller->GetControlRotation();
		const FRotator YawRotation(0, Rotation.Yaw, 0);

		// get forward vector
		const FVector Direction = FRotationMatrix(YawRotation).GetUnitAxis(EAxis::X);
		AddMovementInput(Direction, Value);
	}
}

void ACookbook_Chapter9Character::MoveRight(float Value)
{
	if ( (Controller != NULL) && (Value != 0.0f) )
	{
		// find out which way is right
		const FRotator Rotation = Controller->GetControlRotation();
		const FRotator YawRotation(0, Rotation.Yaw, 0);
	
		// get right vector 
		const FVector Direction = FRotationMatrix(YawRotation).GetUnitAxis(EAxis::Y);
		// add movement in that direction
		AddMovementInput(Direction, Value);
	}
}

void ACookbook_Chapter9Character::Tick(float DeltaTime)
{
	APlayerController * PController = Cast<APlayerController>(Controller);

	// Any time we cast, we need to check if the variable is valid
	if (PController != NULL)
	{
		//  If Q is pressed, save the game
		if (PController->WasInputKeyJustPressed(EKeys::Q))
		{
			SaveMyGameFile();
		}
		// Otherwise, if E is pressed we will load
		else if (PController->WasInputKeyJustPressed(EKeys::E))
		{
			LoadMyGameFile();
		}
	}
}

void ACookbook_Chapter9Character::SaveMyGameFile()
{
	// Create a save object for us to store values
	UMySaveGame* SaveGameInstance = Cast<UMySaveGame>(UGameplayStatics::CreateSaveGameObject(UMySaveGame::StaticClass()));

	// Set the player's current location and rotation
	SaveGameInstance->PlayerPosition = GetActorLocation();
	SaveGameInstance->PlayerRotation = GetActorRotation();

	// Saves the new save file into the first save slot (0) with a name of "SaveSlot"
	UGameplayStatics::SaveGameToSlot(SaveGameInstance, "SaveSlot", 0);

	if (GEngine)
	{
		// Notify the player that we saved
		GEngine->AddOnScreenDebugMessage(-1, 10.0f, FColor::Yellow, "Game Saved!");
	}
}

void ACookbook_Chapter9Character::LoadMyGameFile()
{
	// Create a save object for us to store values to load
	UMySaveGame* LoadGameInstance = Cast<UMySaveGame>(UGameplayStatics::CreateSaveGameObject(UMySaveGame::StaticClass()));

	// Loads the save slot we created previously
	LoadGameInstance = Cast<UMySaveGame>(UGameplayStatics::LoadGameFromSlot("SaveSlot", 0));

	// Set the player's location and rotation to what we saved
	SetActorLocationAndRotation(LoadGameInstance->PlayerPosition, LoadGameInstance->PlayerRotation);

	if (GEngine)
	{
		// Notify the player that we loaded
		GEngine->AddOnScreenDebugMessage(-1, 10.0f, FColor::Yellow, "Loaded!");
	}
}