#include <windows.h>
#include "Model.h"
#include "Sprite.h"
#include "Input.h"
#include "glut.h"

enum GameState
{
	GS_Splash,
	GS_Loading,
	GS_Menu,
	GS_Credits,
	GS_Running,
	GS_NextLevel,
	GS_Paused,
	GS_GameOver,
};

HINSTANCE hInstance = NULL;
HDC hDC = NULL;
HGLRC hRC = NULL;
HWND hWnd = NULL;
bool fullscreen = false;
int screenHeight = 768.0f;
int screenWidth = 1024.0f;
float fRotate = 1.0f;

Model* ship;
std::vector<Model*> asteroids;

Input* inputManager;

GameState gameState;

float splashDisplayTimer;
float splashDisplayThreshold;

Sprite* splashScreen;
Sprite* menuScreen;
Sprite* creditsScreen;
Sprite* playButton;
Sprite* creditsButton;
Sprite* exitButton;
Sprite* menuButton;
Sprite* gameOverScreen;
Sprite* replayButton;

float uiTimer;
const float UI_THRESHOLD = 0.1f;

GLuint	fontBase;
int score;
int speed;
int missionTime;
int asteroidsHit;
int maximumSpeed;


LRESULT CALLBACK WndProc(HWND, UINT, WPARAM, LPARAM);

void ReSizeGLScene(const GLsizei p_width, const GLsizei p_height)
{
	GLsizei h = p_height;
	GLsizei w = p_width;
	if (h == 0)
	{
		h = 1;
	}
	glViewport(0, 0, w, h);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0f, (GLfloat)w / (GLfloat)h, 0.1f, 100.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

const bool InitGL()
{
	glShadeModel(GL_SMOOTH);
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	glClearDepth(1.0f);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
	return true;
}

GLvoid KillGLWindow(GLvoid)
{
	if (fullscreen)
	{
		ChangeDisplaySettings(NULL, 0);
		ShowCursor(TRUE);
	}
	if (hRC)
	{
		wglMakeCurrent(NULL, NULL);
		wglDeleteContext(hRC);
		hRC = NULL;
	}
	if (hDC)
	{
		ReleaseDC(hWnd, hDC);
		hDC = NULL;
	}
	if (hWnd)
	{
		DestroyWindow(hWnd);
		hWnd = NULL;
	}
	UnregisterClass("OpenGL", hInstance);
	hInstance = NULL;
}

const bool CreateGLWindow(const char* p_title, const int p_width, const int p_height, const int p_bits, const bool p_fullscreenflag)
{
	GLuint PixelFormat;
	WNDCLASS wc;
	DWORD dwExStyle;
	DWORD dwStyle;
	RECT WindowRect;
	WindowRect.left = (long)0;
	WindowRect.right = (long)p_width;
	WindowRect.top = (long)0;
	WindowRect.bottom = (long)p_height;
	fullscreen = p_fullscreenflag;
	GLfloat screen_height = (GLfloat)p_height;
	GLfloat screen_width = (GLfloat)p_width;
	hInstance = GetModuleHandle(NULL);
	wc.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
	wc.lpfnWndProc = (WNDPROC)WndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hInstance = hInstance;
	wc.hIcon = LoadIcon(NULL, IDI_WINLOGO);
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = NULL;
	wc.lpszMenuName = NULL;
	wc.lpszClassName = "OpenGL";
	RegisterClass(&wc);
	if (fullscreen)
	{
		DEVMODE dmScreenSettings;
		memset(&dmScreenSettings, 0, sizeof(dmScreenSettings));
		dmScreenSettings.dmSize = sizeof(dmScreenSettings);
		dmScreenSettings.dmPelsWidth = p_width;
		dmScreenSettings.dmPelsHeight = p_height;
		dmScreenSettings.dmBitsPerPel = p_bits;
		dmScreenSettings.dmFields = DM_BITSPERPEL | DM_PELSWIDTH | DM_PELSHEIGHT;
		ChangeDisplaySettings(&dmScreenSettings, CDS_FULLSCREEN);
	}
	if (fullscreen)
	{
		dwExStyle = WS_EX_APPWINDOW;
		dwStyle = WS_POPUP;
		ShowCursor(false);
	}
	else
	{
		dwExStyle = WS_EX_APPWINDOW | WS_EX_WINDOWEDGE;
		dwStyle = WS_OVERLAPPEDWINDOW;
	}
	AdjustWindowRectEx(&WindowRect, dwStyle, FALSE, dwExStyle);
	hWnd = CreateWindowEx(dwExStyle, "OpenGL", p_title,
		dwStyle | WS_CLIPSIBLINGS | WS_CLIPCHILDREN,
		0, 0, WindowRect.right - WindowRect.left, WindowRect.bottom -
		WindowRect.top,
		NULL, NULL, hInstance, NULL);
	static PIXELFORMATDESCRIPTOR pfd =
	{
		sizeof(PIXELFORMATDESCRIPTOR),
		1,
		PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER,
		PFD_TYPE_RGBA, p_bits,
		0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0,
		16, 0, 0,
		PFD_MAIN_PLANE,
		0, 0, 0, 0
	};
	hDC = GetDC(hWnd);
	PixelFormat = ChoosePixelFormat(hDC, &pfd);
	SetPixelFormat(hDC, PixelFormat, &pfd);
	hRC = wglCreateContext(hDC);
	wglMakeCurrent(hDC, hRC);
	ShowWindow(hWnd, SW_SHOW);
	SetForegroundWindow(hWnd);
	SetFocus(hWnd);
	ReSizeGLScene(p_width, p_height);
	InitGL();
	return true;
}

GLvoid BuildFont(GLvoid)
{
	HFONT	newFont;
	HFONT	tempFont;

	fontBase = glGenLists(96);
	glColor3f(0.0f, 1.0f, 0.0f);

	tempFont = CreateFont(-26,	// Height
		0,				// Width
		0,				// Escapement
		0,				// Orientation
		FW_BOLD,			// Weight
		FALSE,				// Italic
		FALSE,				// Underline
		FALSE,				// Strikeout
		ANSI_CHARSET,		// Character Set
		OUT_TT_PRECIS,		// Output Precision
		CLIP_DEFAULT_PRECIS,	// Clipping Precision
		ANTIALIASED_QUALITY,	// Output Quality
		FF_DONTCARE | DEFAULT_PITCH, // Family/Pitch
		"Courier New");		// Font Name

	newFont = (HFONT)SelectObject(hDC, tempFont);
	wglUseFontBitmaps(hDC, 32, 96, fontBase);
	SelectObject(hDC, newFont);
	DeleteObject(tempFont);
}

GLvoid KillFont(GLvoid)
{
	glDeleteLists(fontBase, 96);
}

void DrawText(const char* p_text, const float p_x, const float p_y, const float r, const float g, const float b)
{
	glBindTexture(GL_TEXTURE_2D, 0);
	glColor3f(r, g, b);

	glRasterPos2f(p_x, p_y);
	if (p_text != NULL)
	{
		glPushAttrib(GL_LIST_BIT);
		glListBase(fontBase - 32);
		glCallLists(strlen(p_text), GL_UNSIGNED_BYTE, p_text);
		glPopAttrib();
	}
	glColor3f(1.0f, 1.0f, 1.0f);

}

void DrawUi()
{
	float startY = screenHeight - 50.0f;
	float x1 = 50.0f;
	float x2 = screenWidth / 2.0f - 50.0f;
	float x3 = screenWidth - 250.0f;
	
	char scoreText[50];
	char speedText[50];
	char missionTimeText[50];

	sprintf_s(scoreText, 50, "Score: %i", score);
	sprintf_s(speedText, 50, "Speed: %i", speed);
	sprintf_s(missionTimeText, 50, "Time: %f", missionTime / 100.0f);
	DrawText(scoreText, x1, startY, 0.0f, 1.0f, 0.0f);
	DrawText(speedText, x2, startY, 0.0f, 1.0f, 0.0f);
	DrawText(missionTimeText, x3, startY, 0.0f, 1.0f, 0.0f);

}

void DrawStats()
{
	float startX = screenWidth - screenWidth / 2.5f;
	float startY = 275.0f;
	float spaceY = 30.0f;

	char asteroidsHitText[50];
	char maximumSpeedText[50];
	char scoreText[50];
	char missionTimeText[50];
	sprintf_s(asteroidsHitText, 50, "Asteroids Hit: %i", asteroidsHit);
	sprintf_s(maximumSpeedText, 50, "Maximum Speed: %i", maximumSpeed);
	sprintf_s(scoreText, 50, "Score: %i", score);
	sprintf_s(missionTimeText, 50, "Time: %f", missionTime / 100.0f);
	DrawText(asteroidsHitText, startX, startY, 0.0f, 1.0f, 0.0f);
	DrawText(maximumSpeedText, startX, startY + spaceY, 0.0f, 1.0f, 0.0f);
	DrawText(scoreText, startX, startY + spaceY * 2.0f, 0.0f, 1.0f, 0.0f);
	DrawText(missionTimeText, startX, startY + spaceY * 3.0f, 0.0f, 1.0f, 0.0f);
}
void DrawCredits()
{
	float startX = screenWidth - screenWidth / 2.5f;
	float startY = 300.0f;
	float spaceY = 30.0f;
	DrawText("Robert Madsen", startX, startY, 0.0f, 1.0f, 0.0f);
	DrawText("Author", startX, startY + spaceY, 0.0f, 1.0f, 0.0f);
}

void Enable2D()
{
	glColor3f(1.0f, 1.0f, 1.0f);
	glEnable(GL_TEXTURE_2D);
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glOrtho(0, screenWidth, screenHeight, 0, 0, 1);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glPushAttrib(GL_DEPTH_BUFFER_BIT);
	glDisable(GL_DEPTH_TEST);
}

void Disable2D()
{
	glPopAttrib();
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
	glDisable(GL_TEXTURE_2D);
}

void LoadSplash()
{
	gameState = GameState::GS_Splash;
	splashScreen = new Sprite(1);
	splashScreen->SetFrameSize(screenWidth, screenHeight);
	splashScreen->SetNumberOfFrames(1);
	splashScreen->AddTexture("resources/splash.png", false);
	splashScreen->IsActive(true);
	splashScreen->IsVisible(true);
	splashScreen->SetPosition(0.0f, 0.0f);
}

void CheckCollisions()
{
	bool collision = false;
	for (int i = 0; i < asteroids.size(); i++)
	{
		Model* item = asteroids[i];
		collision = ship->CollidedWith(item);
		if (collision)
		{
			item->IsCollideable(false);
			item->IsVisible(false);
			score++;
			asteroidsHit++;
		}
	}
}

void StartGame()
{
	inputManager = new Input(hWnd);

	//Ship
	Model::Color c(0.0f, 0.0f, 1.0f);
	ship = new Model("ship.obj", c);
	Vec3 rotation(90.0f, 0.0f, 0.0f);
	ship->SetBaseRotation(rotation);
	ship->IsShip(true);
	ship->SetVelocity(1.0f);
	//Asteroid 1
	c.r = 1.0f;
	c.g = 0.0f;
	c.b = 0.0f;
	Model* asteroid = new Model("asteroid.obj", c);
	Vec3 position(0.0f, 0.0f, -10.0f);
	asteroid->SetPosition(position);
	asteroids.push_back(asteroid);
	//Asteroid 2
	c.r = 0.0f;
	c.g = 1.0f;
	c.b = 0.0f;
	asteroid = new Model("asteroid.obj", c);
	position.x = 5.0f;
	position.y = 0.0f;
	position.z = -15.0f;
	asteroid->SetPosition(position);
	asteroids.push_back(asteroid);
	//Asteroid 3
	c.r = 0.0f;
	c.g = 1.0f;
	c.b = 1.0f;
	asteroid = new Model("asteroid.obj", c);
	position.x = 5.0f;
	position.y = 5.0f;
	position.z = -20.0f;
	asteroid->SetPosition(position);
	asteroids.push_back(asteroid);

	LoadSplash();
	uiTimer = 0.0f;
	splashDisplayTimer = 0.0f;
	splashDisplayThreshold = 5.0f;

	score = 0;
	speed = 1.0f;
	maximumSpeed = 0;
	asteroidsHit = 0;
	missionTime = 0;
}

void ProcessInput(const float p_deltaTime)
{
	Vec3 rotation;
	inputManager->Update(p_deltaTime);
	Input::Command command = inputManager->GetCommand();
	switch (command)
	{
	case Input::CM_STOP:
	{
		if (ship->GetVelocity() > 0.0f)
		{
			ship->SetVelocity(0.0f);
		}
		else
		{
			ship->SetVelocity(1.0f);
		}
	}
	break;
	case Input::CM_DOWN:
	{
		rotation = ship->GetHeadingRotation();
		rotation.x += -1.0f;
		if (rotation.x < 0.0f)
		{
			rotation.x = 359.0f;
		}
		if (rotation.x < 359.0f && rotation.x > 180.0f)
		{
			if (rotation.x < 315.0f)
			{
				rotation.x = 315.0f;
			}
		}
		ship->SetHeadingRotation(rotation);
	}
	break;
	case Input::CM_UP:
	{
		rotation = ship->GetHeadingRotation();
		rotation.x += 1.0f;
		if (rotation.x > 359.0f)
		{
			rotation.x = 0.0f;
		}
		if (rotation.x > 0.0f && rotation.x < 180.0f)
		{
			if (rotation.x > 45.0f)
			{
				rotation.x = 45.0f;
			}
		}
		ship->SetHeadingRotation(rotation);
	}
	break;
	case Input::CM_LEFT:
	{
		rotation = ship->GetHeadingRotation();
		rotation.z += 1.0f;
		if (rotation.z > 359.0f)
		{
			rotation.z = 0.0f;
		}
		if (rotation.z > 0.0f && rotation.z < 180.0f)
		{
			if (rotation.z > 45.0f)
			{
				rotation.z = 45.0f;
			}
		}
		ship->SetHeadingRotation(rotation);
	}
	break;
	case Input::CM_RIGHT:
	{
		rotation = ship->GetHeadingRotation();
		rotation.z += -1.0f;
		if (rotation.z < 0.0f)
		{
			rotation.z = 359.0f;
		}
		if (rotation.z < 359.0f && rotation.z > 180.0f)
		{
			if (rotation.z < 315.0f)
			{
				rotation.z = 315.0f;
			}
		}
		ship->SetHeadingRotation(rotation);
	}
	break;
	case Input::Command::CM_UI:
	{
		if (playButton->IsClicked())
		{
			playButton->IsClicked(false);
			exitButton->IsActive(false);
			playButton->IsActive(false);
			creditsButton->IsActive(false);
			gameState = GameState::GS_Running;
		}
		if (creditsButton->IsClicked())
		{
			creditsButton->IsClicked(false);
			exitButton->IsActive(false);
			playButton->IsActive(false);
			creditsButton->IsActive(false);
			gameState = GameState::GS_Credits;
		}
		if (menuButton->IsClicked())
		{
			menuButton->IsClicked(false);
			exitButton->IsActive(true);
			playButton->IsActive(true);
			menuButton->IsActive(false);
			switch (gameState)
			{
			case GameState::GS_Credits:
			{
				gameState = GameState::GS_Menu;
			}
			break;
			case GameState::GS_GameOver:
			{
				StartGame();
			}
			break;
			}
		}
		if (exitButton->IsClicked())
		{
			playButton->IsClicked(false);
			exitButton->IsActive(false);
			playButton->IsActive(false);
			creditsButton->IsActive(false);
			PostQuitMessage(0);
		}
	}
	break;
	}
}

void DrawCube()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glTranslatef(0.0f, 0.0f, -7.0f);
	glRotatef(fRotate, 1.0f, 1.0f, 1.0f);
	glBegin(GL_QUADS);
	glColor3f(0.0f, 1.0f, 0.0f);
	glVertex3f(1.0f, 1.0f, -1.0f); glVertex3f(-1.0f, 1.0f, -1.0f);
	glVertex3f(-1.0f, 1.0f, 1.0f); glVertex3f(1.0f, 1.0f, 1.0f);
	glColor3f(1.0f, 0.5f, 0.0f);
	glVertex3f(1.0f, -1.0f, 1.0f); glVertex3f(-1.0f, -1.0f, 1.0f);
	glVertex3f(-1.0f, -1.0f, -1.0f); glVertex3f(1.0f, -1.0f, -1.0f);
	glColor3f(1.0f, 0.0f, 0.0f);
	glVertex3f(1.0f, 1.0f, 1.0f); glVertex3f(-1.0f, 1.0f, 1.0f);
	glVertex3f(-1.0f, -1.0f, 1.0f); glVertex3f(1.0f, -1.0f, 1.0f);
	glColor3f(1.0f, 1.0f, 0.0f);
	glVertex3f(1.0f, -1.0f, -1.0f); glVertex3f(-1.0f, -1.0f, -1.0f);
	glVertex3f(-1.0f, 1.0f, -1.0f); glVertex3f(1.0f, 1.0f, -1.0f);
	glColor3f(0.0f, 0.0f, 1.0f);
	glVertex3f(-1.0f, 1.0f, 1.0f); glVertex3f(-1.0f, 1.0f, -1.0f);
	glVertex3f(-1.0f, -1.0f, -1.0f); glVertex3f(-1.0f, -1.0f, 1.0f);
	glColor3f(1.0f, 0.0f, 1.0f);
	glVertex3f(1.0f, 1.0f, -1.0f); glVertex3f(1.0f, 1.0f, 1.0f);
	glVertex3f(1.0f, -1.0f, 1.0f); glVertex3f(1.0f, -1.0f, -1.0f);
	glEnd();
	fRotate -= 0.05f;
}

const bool LoadTextures()
{
	menuScreen = new Sprite(1);
	menuScreen->SetFrameSize(screenWidth, screenHeight);
	menuScreen->SetNumberOfFrames(1);
	menuScreen->AddTexture("resources/mainmenu.png", false);
	menuScreen->IsActive(true);
	menuScreen->IsVisible(true);
	menuScreen->SetPosition(0.0f, 0.0f);
	playButton = new Sprite(1);
	playButton->SetFrameSize(75.0f, 38.0f);
	playButton->SetNumberOfFrames(1);
	playButton->SetPosition(690.0f, 300.0f);
	playButton->AddTexture("resources/playButton.png");
	playButton->IsVisible(true);
	playButton->IsActive(false);
	inputManager->AddUiElement(playButton);
	creditsButton = new Sprite(1);
	creditsButton->SetFrameSize(75.0f, 38.0f);
	creditsButton->SetNumberOfFrames(1);
	creditsButton->SetPosition(690.0f, 350.0f);
	creditsButton->AddTexture("resources/creditsButton.png");
	creditsButton->IsVisible(true);
	creditsButton->IsActive(false);
	inputManager->AddUiElement(creditsButton);
	exitButton = new Sprite(1);
	exitButton->SetFrameSize(75.0f, 38.0f);
	exitButton->SetNumberOfFrames(1);
	exitButton->SetPosition(690.0f, 500.0f);
	exitButton->AddTexture("resources/exitButton.png");
	exitButton->IsVisible(true);
	exitButton->IsActive(false);
	inputManager->AddUiElement(exitButton);
	creditsScreen = new Sprite(1);
	creditsScreen->SetFrameSize(screenWidth, screenHeight);
	creditsScreen->SetNumberOfFrames(1);
	creditsScreen->AddTexture("resources/credits.png", false);
	creditsScreen->IsActive(true);
	creditsScreen->IsVisible(true);
	menuButton = new Sprite(1);
	menuButton->SetFrameSize(75.0f, 38.0f);
	menuButton->SetNumberOfFrames(1);
	menuButton->SetPosition(690.0f, 400.0f);
	menuButton->AddTexture("resources/menuButton.png");
	menuButton->IsVisible(true);
	menuButton->IsActive(false);
	inputManager->AddUiElement(menuButton);
	gameOverScreen = new Sprite(1);
	gameOverScreen->SetFrameSize(screenWidth, screenHeight);
	gameOverScreen->SetNumberOfFrames(1);
	gameOverScreen->AddTexture("resources/gameover.png", false);
	gameOverScreen->IsActive(true);
	gameOverScreen->IsVisible(true);
	replayButton = new Sprite(1);
	replayButton->SetFrameSize(75.0f, 38.0f);
	replayButton->SetNumberOfFrames(1);
	replayButton->SetPosition(690.0f, 400.0f);
	replayButton->AddTexture("resources/replayButton.png");
	replayButton->IsVisible(true);
	replayButton->IsActive(false);
	inputManager->AddUiElement(replayButton);
	return true;
}



void Update(const float p_deltaTime)
{
	switch (gameState)
	{
	case GameState::GS_Splash:
	case GameState::GS_Loading:
	{
		splashScreen->Update(p_deltaTime);
		splashDisplayTimer += p_deltaTime;
		if (splashDisplayTimer > splashDisplayThreshold)
		{
			gameState = GameState::GS_Menu;
		}
	}
	break;
	case GameState::GS_Menu:
	{
		menuScreen->Update(p_deltaTime);
		playButton->IsActive(true);
		creditsButton->IsActive(true);
		exitButton->IsActive(true);
		playButton->Update(p_deltaTime);
		creditsButton->Update(p_deltaTime);
		exitButton->Update(p_deltaTime);
		inputManager->Update(p_deltaTime);
		ProcessInput(p_deltaTime);
	}
	break;
	case GameState::GS_Credits:
	{
		creditsScreen->Update(p_deltaTime);
		menuButton->IsActive(true);
		menuButton->Update(p_deltaTime);
		inputManager->Update(p_deltaTime);
		ProcessInput(p_deltaTime);
	}
	break;
	case GameState::GS_Running:
	{
		inputManager->Update(p_deltaTime);
		ProcessInput(p_deltaTime);
		ship->Update(p_deltaTime);
		ship->SetVelocity(ship->GetVelocity() + ship->GetVelocity()*p_deltaTime/10.0f);
		speed = ship->GetVelocity() * 1000;
		if (maximumSpeed < speed)
		{
			maximumSpeed = speed;
		}
		missionTime = missionTime + p_deltaTime * 100.0f;
		CheckCollisions();
		if (ship->GetPosition().z > 10.0f)
		{
			gameState = GS_GameOver;
			menuButton->IsActive(true);
			gameOverScreen->IsActive(true);
		}
	}
	break;
	case GameState::GS_GameOver:
	{
		gameOverScreen->Update(p_deltaTime);
		replayButton->IsActive(true);
		replayButton->Update(p_deltaTime);
		exitButton->IsActive(true);
		exitButton->Update(p_deltaTime);
		inputManager->Update(p_deltaTime);
		ProcessInput(p_deltaTime);
	}
	break;
	}
}


void Render2D()
{
	Enable2D();
	switch (gameState)
	{
	case GameState::GS_Loading:
	{
		splashScreen->Render();
	}
	break;
	case GameState::GS_Menu:
	{
		menuScreen->Render();
		playButton->Render();
		creditsButton->Render();
		exitButton->Render();
	}
	break;
	case GameState::GS_Credits:
	{
		creditsScreen->Render();
		menuButton->Render();
		DrawCredits();
	}
	break;
	case GameState::GS_Running:
	{
		DrawUi();
	}
	break;
	case GameState::GS_Splash:
	{
		splashScreen->Render();
	}
		break;
	case GameState::GS_GameOver:
	{
		gameOverScreen->Render();
		DrawStats();
		menuButton->Render();
	}
	break;
	}
	Disable2D();
}

void Render3D()
{
	if (gameState == GS_Running)
	{
		for (unsigned int i = 0; i < asteroids.size(); i++)
		{
			asteroids[i]->Render();
		}
		ship->Render();
	}
}


void Render()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	Render3D();
	Render2D();
	SwapBuffers(hDC);
}

void EndGame()
{
}
void GameLoop(const float p_deltatTime)
{
	if (gameState == GameState::GS_Splash)
	{
		BuildFont();
		LoadTextures();
		gameState = GameState::GS_Loading;
	}
	Update(p_deltatTime);
	Render();
}
int APIENTRY WinMain(_In_ HINSTANCE hInstance,
	_In_opt_ HINSTANCE hPrevInstance,
	_In_ LPTSTR lpCmdLine,
	_In_ int nCmdShow)
{
	MSG msg;
	bool done = false;
	if (!CreateGLWindow("SpaceRacer3D", screenWidth, screenHeight, 16, false))
	{
		return false;
	}
	StartGame();
	int previousTime = glutGet(GLUT_ELAPSED_TIME);
	while (!done)
	{
		if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
		{
			if (msg.message == WM_QUIT)
			{
				done = true;
			}
			else
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}
		else
		{
			int currentTime = glutGet(GLUT_ELAPSED_TIME);
			float deltaTime = (float)(currentTime - previousTime) / 1000;
			previousTime = currentTime;
			GameLoop(deltaTime);
		}
	}
	EndGame();
	return (int)msg.wParam;
}

LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam,
	LPARAM lParam)
{
	switch (message)
	{
	case WM_DESTROY:
		PostQuitMessage(0);
		break;
	case WM_SIZE:
		ReSizeGLScene(LOWORD(lParam), HIWORD(lParam));
		return 0;
	default:
		return DefWindowProc(hWnd, message, wParam, lParam);
	}
	return false;
}

