#include "Model.h"

Model::Model(const char* p_filepath, const Color p_color)
{
	m_filepath = p_filepath;
	m_loaded = LoadObj(m_filepath, m_vertices, m_normals, m_primitive);
	SetPosition(0.0f, 0.0f, 0.0f);
	SetHeading(0.0f, 0.0f, 0.0f);
	SetHeadingRotation(0.0f, 0.0f, 0.0f);
	SetBaseRotation(0.0f, 0.0f, 0.0f);
	IsShip(false);
	SetVelocity(0.0f);
	SetColor(p_color.r, p_color.g, p_color.b);
	SetRadius(1.0f);
	IsCollideable(true);
	IsVisible(true);

}
Model::~Model()
{
	m_vertices.clear();
	m_normals.clear();
}

float Deg2Rad(const float p_degrees)
{
	return p_degrees * (M_PI / 180.0f);
}

void Model::Update(const float p_deltaTime)
{
	Vec3 targetRotation = GetHeadingRotation();
	Vec3 currentPosition = GetPosition();
	Vec3 targetPosition = GetPosition();

	float distance = m_velocity * p_deltaTime;
	Vec3 deltaPosition;

	deltaPosition.y = cos(Deg2Rad(targetRotation.z)) * distance;
	deltaPosition.x = -sin(Deg2Rad(targetRotation.z)) * distance;
	deltaPosition.z = sin(Deg2Rad(targetRotation.x)) * distance;

	targetPosition.x += deltaPosition.x;
	targetPosition.y += deltaPosition.y;
	targetPosition.z += deltaPosition.z;
	SetPosition(targetPosition);
}


void Model::Render()
{
	if (IsVisible())
	{
		glRotatef(-m_baseRotation.x, 1.0f, 0.0f, 0.0f);
		glRotatef(-m_baseRotation.y, 0.0f, 1.0f, 0.0f);
		glRotatef(-m_baseRotation.z, 0.0f, 0.0f, 1.0f);

		Vec3 targetRotation = GetHeadingRotation();
		Vec3 currentPosition = GetPosition();

		if (m_IsShip)
		{
			glPushMatrix();
			glLoadIdentity();
			glRotatef(targetRotation.x, 1.0f, 0.0f, 0.0f);
			glRotatef(targetRotation.y, 0.0f, 1.0f, 0.0f);
			glRotatef(targetRotation.z, 0.0f, 0.0f, 1.0f);
			GLfloat matrix[16];
			glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
			glPopMatrix();
			glTranslatef(currentPosition.x, currentPosition.y, currentPosition.z);
			glMultMatrixf(matrix);
		}

		switch (m_primitive)
		{
		case Primitive::Quads:
			glBegin(GL_QUADS);
			break;
		case Primitive::Triangles:
			glBegin(GL_TRIANGLES);
			break;
		}
		glColor3f(m_color.r, m_color.g, m_color.b);
		for (unsigned int i = 0; i < m_vertices.size(); i++)
		{
			if (m_IsShip)
			{
				glVertex3f(m_vertices[i].x, m_vertices[i].y, m_vertices[i].z);
			}
			else
			{
				glVertex3f(m_vertices[i].x + m_position.x, m_vertices[i].y + m_position.y, m_vertices[i].z + m_position.z);
			}
		}
		glEnd();
	}
}

void Model::SetPosition(const float p_x, const float p_y, const float p_z)
{
	m_position.x = p_x;
	m_position.y = p_y;
	m_position.z = p_z;
}
void Model::SetPosition(const Vec3 p_position)
{
	m_position.x = p_position.x;
	m_position.y = p_position.y;
	m_position.z = p_position.z;
}
const Vec3 Model::GetPosition() const
{
	return m_position;
}

void Model::SetHeading(const float p_x, const float p_y, const float
	p_z)
{
	m_heading.x = p_x;
	m_heading.y = p_y;
	m_heading.z = p_z;
}
void Model::SetHeading(const Vec3 p_heading)
{
	m_heading.x = p_heading.x;
	m_heading.y = p_heading.y;
	m_heading.z = p_heading.z;
}
const Vec3 Model::GetHeading() const
{
	return m_heading;
}

void Model::SetColor(const float p_red, const float p_green, const
	float p_blue)
{
	m_color.r = p_red;
	m_color.g = p_green;
	m_color.b = p_blue;
}
void Model::SetColor(const Color p_color)
{
	m_color.r = p_color.r;
	m_color.g = p_color.g;
	m_color.b = p_color.b;
}

void Model::SetVelocity(const float p_velocity)
{
	m_velocity = p_velocity;
}
const float Model::GetVelocity() const
{
	return m_velocity;
}

void Model::SetBaseRotation(const float p_x, const float p_y, const
	float p_z)
{
	m_baseRotation.x = p_x;
	m_baseRotation.y = p_y;
	m_baseRotation.z = p_z;
}
void Model::SetBaseRotation(const Vec3 p_rotation)
{
	m_baseRotation.x = p_rotation.x;
	m_baseRotation.y = p_rotation.y;
	m_baseRotation.z = p_rotation.z;
}
const Vec3 Model::GetBaseRotation() const
{
	return m_baseRotation;
}

void Model::SetHeadingRotation(const float p_x, const float p_y, const
	float p_z)
{
	m_headingRotation.x = p_x;
	m_headingRotation.y = p_y;
	m_headingRotation.z = p_z;
}
void Model::SetHeadingRotation(const Vec3 p_rotation)
{
	m_headingRotation.x = p_rotation.x;
	m_headingRotation.y = p_rotation.y;
	m_headingRotation.z = p_rotation.z;
}
const Vec3 Model::GetHeadingRotation() const
{
	return m_headingRotation;
}

const bool Model::IsShip()
{
	return m_IsShip;
}
void Model::IsShip(const bool p_IsShip)
{
	m_IsShip = p_IsShip;
}

const bool Model::IsCollideable()
{
	return m_collideable;
}
void Model::IsCollideable(const bool p_collideable)
{
	m_collideable = p_collideable;
}
const bool Model::CollidedWith(Model* p_target)
{
	if (p_target->IsCollideable() && this->IsCollideable())
	{
		const Vec3 p1 = this->GetCenter();
		const Vec3 p2 = p_target->GetCenter();
		float y = p2.y - p1.y;
		float x = p2.x - p1.x;
		float z = p2.z - p1.z;
		float d = x*x + y*y + z*z;
		float r1 = this->GetRadius() * this->GetRadius();
		float r2 = p_target->GetRadius() * p_target->GetRadius();
		if (d <= r1 + r2)
		{
			return true;
		}
	}
	return false;
}
const Vec3 Model::GetCenter() const
{
	Vec3 center;
	center = GetPosition();
	if (m_IsShip)
	{
		center.z = -m_position.y;
		center.x = m_position.x;
		center.y = m_position.z;
	}
	return center;
}
void Model::SetRadius(const float p_radius)
{
	m_radius = p_radius;
}
const float Model::GetRadius() const
{
	return m_radius;
}

