// RoboRacer2D.cpp : Defines the entry point for the application.
//

#include "stdafx.h"
#include "RoboRacer2D.h"
#include <Windows.h>

#include <gl\GL.h>
#include <gl\GLU.h>
#include "glut.h"

#include "Sprite.h"
#include "Input.h"

#include  <stdio.h>
#include <time.h>

#define MAX_LOADSTRING 100

enum GameState
{
	GS_Splash,
	GS_Loading,
	GS_Menu,
	GS_Credits,
	GS_Running,
	GS_NextLevel,
	GS_Paused,
	GS_GameOver,
};

// Global Variables:
HINSTANCE hInst;                                // current instance
WCHAR szTitle[MAX_LOADSTRING];                  // The title bar text
WCHAR szWindowClass[MAX_LOADSTRING];            // the main window class name
HDC			hDC = NULL;
HGLRC		hRC = NULL;
HWND		hWnd = NULL;


bool fullscreen = false;
GLfloat screen_height;
GLfloat screen_width;

//Game variables
Sprite* robot_left;
Sprite* robot_right;
Sprite* robot_right_strip;
Sprite* robot_left_strip;
Sprite* background;
Sprite* player;

Sprite* pauseButton;
Sprite* resumeButton;

Sprite* pickup;
Sprite* enemy;

Sprite* splashScreen;
Sprite* menuScreen;
Sprite* creditsScreen;
Sprite* nextLevelScreen;
Sprite* continueButton;
Sprite* gameOverScreen;
Sprite* replayButton;

Sprite* playButton;
Sprite* creditsButton;
Sprite* exitButton;
Sprite* menuButton;

Input* inputManager;

GameState m_gameState;

float uiTimer;
const float UI_THRESHOLD = 0.2f;

float pickupSpawnThreshold;
float pickupSpawnTimer;

float enemySpawnThreshold;
float enemySpawnTimer;

float splashDisplayTimer;
float splashDisplayThreshold;

float levelTimer;
float levelMaxTime;
float pickupSpawnAdjustment;
int pickupsReceived;
int pickupsThreshold;
int enemiesHit;

GLuint fontBase;


// Forward declarations of functions included in this code module:
ATOM                MyRegisterClass(HINSTANCE hInstance);
BOOL                InitInstance(HINSTANCE, int);
LRESULT CALLBACK    WndProc(HWND, UINT, WPARAM, LPARAM);
INT_PTR CALLBACK    About(HWND, UINT, WPARAM, LPARAM);

void ReSizeGLScene(const GLsizei p_width, const GLsizei p_height)
{
	GLsizei h = p_height;
	GLsizei w = p_width;
	if (h == 0) h = 1;
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, w, h, 0, 0, 1);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

}
const bool InitGL()
{
	glEnable(GL_TEXTURE_2D);
	glShadeModel(GL_SMOOTH);
	glClearColor(0.0f, 0.0f, 0.0f, 0.5f);
	glDisable(GL_DEPTH_TEST);
	return true;
}

const bool CreateGLWindow(const char* p_title, const int p_width, const int p_height, const int p_bits, const bool p_fullscreenflag)
{
	GLuint		PixelFormat;
	WNDCLASS	wc;
	DWORD		dwExStyle;
	DWORD		dwStyle;
	RECT		WindowRect;
	WindowRect.left = (long)0;
	WindowRect.right = (long)p_width;
	WindowRect.top = (long)0;
	WindowRect.bottom = (long)p_height;
	fullscreen = p_fullscreenflag;
	screen_height = (GLfloat)p_height;
	screen_width = (GLfloat)p_width;
	hInst = GetModuleHandle(NULL);
	wc.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
	wc.lpfnWndProc = (WNDPROC)WndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hInstance = hInst;
	wc.hIcon = LoadIcon(NULL, IDI_WINLOGO);
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = NULL;
	wc.lpszMenuName = NULL;
	wc.lpszClassName = "OpenGL";
	if (!RegisterClass(&wc))
	{
		return false;
	}
	if (fullscreen)
	{
		DEVMODE dmScreenSettings;
		memset(&dmScreenSettings, 0, sizeof(dmScreenSettings));
		dmScreenSettings.dmSize = sizeof(dmScreenSettings);
		dmScreenSettings.dmPelsWidth = p_width;
		dmScreenSettings.dmPelsHeight = p_height;
		dmScreenSettings.dmBitsPerPel = p_bits;
		dmScreenSettings.dmFields = DM_BITSPERPEL | DM_PELSWIDTH | DM_PELSHEIGHT;

		if (ChangeDisplaySettings(&dmScreenSettings, CDS_FULLSCREEN) != DISP_CHANGE_SUCCESSFUL)
		{
			fullscreen = false;
		}
	}
	if (fullscreen)
	{
		dwExStyle = WS_EX_APPWINDOW;
		dwStyle = WS_POPUP;
		ShowCursor(false);
	}
	else
	{
		dwExStyle = WS_EX_APPWINDOW | WS_EX_WINDOWEDGE;
		dwStyle = WS_OVERLAPPEDWINDOW;
	}
	AdjustWindowRectEx(&WindowRect, dwStyle, FALSE, dwExStyle);

	if (!(hWnd = CreateWindowEx(dwExStyle,
		"OpenGL",
		p_title,
		dwStyle |
		WS_CLIPSIBLINGS |
		WS_CLIPCHILDREN,
		0, 0,
		WindowRect.right - WindowRect.left,
		WindowRect.bottom - WindowRect.top,
		NULL,
		NULL,
		hInst,
		NULL)))
	{
		return false;
	}
	static	PIXELFORMATDESCRIPTOR pfd =
	{
		sizeof(PIXELFORMATDESCRIPTOR),
		1,
		PFD_DRAW_TO_WINDOW |
		PFD_SUPPORT_OPENGL |
		PFD_DOUBLEBUFFER,
		PFD_TYPE_RGBA,
		p_bits,
		0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0,
		16, 0, 0,
		PFD_MAIN_PLANE,
		0, 0, 0, 0
	};
	if (!(hDC = GetDC(hWnd)))
	{
		return false;
	}
	if (!(PixelFormat = ChoosePixelFormat(hDC, &pfd)))
	{
		return false;
	}
	if (!SetPixelFormat(hDC, PixelFormat, &pfd))
	{
		return false;
	}
	if (!(hRC = wglCreateContext(hDC)))
	{
		return false;
	}
	if (!wglMakeCurrent(hDC, hRC))
	{
		return false;
	}
	ShowWindow(hWnd, SW_SHOW);
	SetForegroundWindow(hWnd);
	SetFocus(hWnd);
	ReSizeGLScene(p_width, p_height);
	if (!InitGL())
	{
		return false;
	}
	return true;
}

GLvoid BuildFont(GLvoid)
{
	HFONT newFont;
	HFONT tempFont;
	fontBase = glGenLists(96);
	tempFont = CreateFont(-26, // Height
		0, // Width
		0, // Escapement
		0, // Orientation
		FW_BOLD, // Weight
		FALSE, // Italic
		FALSE, // Underline
		FALSE, // Strikeout
		ANSI_CHARSET, // Character Set
		OUT_TT_PRECIS, // Output Precision
		CLIP_DEFAULT_PRECIS, // Clipping Precision
		ANTIALIASED_QUALITY,// Output Quality
		FF_DONTCARE | DEFAULT_PITCH, // Family/Pitch
		"Courier New"); // Font Name
	newFont = (HFONT)SelectObject(hDC, tempFont);
	wglUseFontBitmaps(hDC, 32, 96, fontBase);
	SelectObject(hDC, newFont);
	DeleteObject(tempFont);
}

GLvoid KillFont(GLvoid)
{
	glDeleteLists(fontBase, 96);
}

void DrawText(const char* p_text, const float p_x, const float p_y, const float r, const float g, const float b)
{
	glBindTexture(GL_TEXTURE_2D, 0);
	glColor3f(r, g, b);

	glRasterPos2f(p_x, p_y);
	if (p_text != NULL)
	{
		glPushAttrib(GL_LIST_BIT);
		glListBase(fontBase - 32);
		glCallLists(strlen(p_text), GL_UNSIGNED_BYTE, p_text);
		glPopAttrib();
	}
	glColor3f(1.0f, 1.0f, 1.0f);
}


void DrawCredits()
{
	float startX = 325.0f;
	float startY = 250.0f;
	float spaceY = 30.0f;
	DrawText("Robert Madsen", startX, startY, 0.0f, 0.0f, 1.0f);
	DrawText("Author", startX, startY + spaceY, 0.0f, 0.0f, 1.0f);
}

void DrawScore()
{
	char score[50];
	sprintf_s(score, 50, "Score: %i", player->GetValue());
	DrawText(score, 350.0f, 25.0f, 0.0f, 0.0f, 1.0f);
}

void DrawStats()
{
	char pickupsStat[50];
	char enemiesStat[50];
	char score[50];
	sprintf_s(pickupsStat, 50, "Enemies Hit: %i", enemiesHit);
	sprintf_s(enemiesStat, 50, "Pickups: %i", pickupsReceived);
	sprintf_s(score, 50, "Score: %i", player->GetValue());
	DrawText(enemiesStat, 350.0f, 270.0f, 0.0f, 0.0f, 1.0f);
	DrawText(pickupsStat, 350.0f, 320.0f, 0.0f, 0.0f, 1.0f);
	DrawText(score, 350.0f, 370.0f, 0.0f, 0.0f, 1.0f);
}

void LoadSplash()
{
	m_gameState = GameState::GS_Splash;
	splashScreen = new Sprite(1);
	splashScreen->SetFrameSize(800.0f, 600.0f);
	splashScreen->SetNumberOfFrames(1);
	splashScreen->AddTexture("resources/splash.png", false);
	splashScreen->IsActive(true);
	splashScreen->IsVisible(true);
}

const bool LoadTextures()
{
	background = new Sprite(1);
	background->SetFrameSize(1877.0f, 600.0f);
	background->SetNumberOfFrames(1);
	background->AddTexture("resources/background.png", false);

	robot_right = new Sprite(4);
	robot_right->SetFrameSize(100.0f, 125.0f);
	robot_right->SetNumberOfFrames(4);
	robot_right->SetPosition(0, screen_height - 130.0f);
	robot_right->AddTexture("resources/robot_right_00.png");
	robot_right->AddTexture("resources/robot_right_01.png");
	robot_right->AddTexture("resources/robot_right_02.png");
	robot_right->AddTexture("resources/robot_right_03.png");

	robot_left = new Sprite(4);
	robot_left->SetFrameSize(100.0f, 125.0f);
	robot_left->SetNumberOfFrames(4);
	robot_left->SetPosition(0, screen_height - 130.0f);
	robot_left->AddTexture("resources/robot_left_00.png");
	robot_left->AddTexture("resources/robot_left_01.png");
	robot_left->AddTexture("resources/robot_left_02.png");
	robot_left->AddTexture("resources/robot_left_03.png");

	robot_right_strip = new Sprite(1);
	robot_right_strip->SetFrameSize(125.0f, 100.0f);
	robot_right_strip->SetNumberOfFrames(4);
	robot_right_strip->SetPosition(0, screen_height - 130.0f);
	robot_right_strip->AddTexture("resources/robot_right_strip.png");

	robot_left_strip = new Sprite(1);
	robot_left_strip->SetFrameSize(125.0f, 100.0f);
	robot_left_strip->SetNumberOfFrames(4);
	robot_right_strip->SetPosition(0, screen_height - 130.0f);
	robot_left_strip->AddTexture("resources/robot_left_strip.png");

	background->IsVisible(true);
	background->IsActive(true);
	background->SetVelocity(-50.0f);

	robot_right->IsActive(true);
	robot_right->IsVisible(true);
	
	player = robot_right;
	player->IsActive(true);
	player->IsVisible(true);
	
	pauseButton = new Sprite(1);
	pauseButton->SetFrameSize(75.0f, 38.0f);
	pauseButton->SetNumberOfFrames(1);
	pauseButton->SetPosition(5.0f, 5.0f);
	pauseButton->AddTexture("resources/pauseButton.png");
	pauseButton->IsVisible(true);
	pauseButton->IsActive(true);
	inputManager->AddUiElement(pauseButton);

	resumeButton = new Sprite(1);
	resumeButton->SetFrameSize(75.0f, 38.0f);
	resumeButton->SetNumberOfFrames(1);
	resumeButton->SetPosition(80.0f, 5.0f);
	resumeButton->AddTexture("resources/resumeButton.png");
	inputManager->AddUiElement(resumeButton);

	Sprite::Rect collision;
	collision.left = 34.0f;
	collision.right = -10.0f;
	collision.top = 0.0f;
	collision.bottom = 0.0f;
	robot_left->SetCollisionRectOffset(collision);
	robot_right->SetCollisionRectOffset(collision);

	pickup = new Sprite(1);
	pickup->SetFrameSize(26.0f, 50.0f);
	pickup->SetNumberOfFrames(1);
	pickup->AddTexture("resources/oil.png");
	pickup->IsVisible(false);
	pickup->IsActive(false);
	pickup->SetValue(50);

	robot_left->IsCollideable(true);
	robot_right->IsCollideable(true);
	pickup->IsCollideable(true);
	

	enemy = new Sprite(1);
	enemy->SetFrameSize(32.0f, 50.0f);
	enemy->SetNumberOfFrames(1);
	enemy->AddTexture("resources/water.png");
	enemy->IsVisible(false);
	enemy->IsActive(false);
	enemy->SetValue(-50);
	enemy->IsCollideable(true);

	Sprite::Point center;
	float radius;
	center.x = robot_right->GetSize().width / 2.0f;
	center.y = robot_right->GetSize().height / 2.0f;
	radius = (center.x + center.y) / 2.0f;
	robot_right->SetCenter(center);
	robot_right->SetRadius(radius);
	robot_left->SetCenter(center);
	robot_left->SetRadius(radius);
	center.x = pickup->GetSize().width / 2.0f;
	float yOffset = (pickup->GetSize().height / 4.0f) * 3.0f;
	center.y = yOffset;
	pickup->SetCenter(center);
	radius = pickup->GetSize().width / 2.0f;
	pickup->SetRadius(radius);

	menuScreen = new Sprite(1);
	menuScreen->SetFrameSize(800.0f, 600.0f);
	menuScreen->SetNumberOfFrames(1);
	menuScreen->AddTexture("resources/mainmenu.png", false);
	menuScreen->IsActive(true);
	menuScreen->IsVisible(true);

	playButton = new Sprite(1);
	playButton->SetFrameSize(75.0f, 38.0f);
	playButton->SetNumberOfFrames(1);
	playButton->SetPosition(390.0f, 300.0f);
	playButton->AddTexture("resources/playButton.png");
	playButton->IsVisible(true);
	playButton->IsActive(false);
	inputManager->AddUiElement(playButton);
	creditsButton = new Sprite(1);
	creditsButton->SetFrameSize(75.0f, 38.0f);
	creditsButton->SetNumberOfFrames(1);
	creditsButton->SetPosition(390.0f, 350.0f);
	creditsButton->AddTexture("resources/creditsButton.png");
	creditsButton->IsVisible(true);
	creditsButton->IsActive(false);
	inputManager->AddUiElement(creditsButton);

	exitButton = new Sprite(1);
	exitButton->SetFrameSize(75.0f, 38.0f);
	exitButton->SetNumberOfFrames(1);
	exitButton->SetPosition(390.0f, 500.0f);
	exitButton->AddTexture("resources/exitButton.png");
	exitButton->IsVisible(true);
	exitButton->IsActive(false);
	inputManager->AddUiElement(exitButton);

	creditsScreen = new Sprite(1);
	creditsScreen->SetFrameSize(800.0f, 600.0f);
	creditsScreen->SetNumberOfFrames(1);
	creditsScreen->AddTexture("resources/credits.png", false);
	creditsScreen->IsActive(false);
	creditsScreen->IsVisible(true);

	menuButton = new Sprite(1);
	menuButton->SetFrameSize(75.0f, 38.0f);
	menuButton->SetNumberOfFrames(1);
	menuButton->SetPosition(390.0f, 400.0f);
	menuButton->AddTexture("resources/menuButton.png");
	menuButton->IsVisible(true);
	menuButton->IsActive(false);
	inputManager->AddUiElement(menuButton);

	nextLevelScreen = new Sprite(1);
	nextLevelScreen->SetFrameSize(800.0f, 600.0f);
	nextLevelScreen->SetNumberOfFrames(1);
	nextLevelScreen->AddTexture("resources/level.png", false);
	nextLevelScreen->IsActive(true);
	nextLevelScreen->IsVisible(true);

	continueButton = new Sprite(1);
	continueButton->SetFrameSize(75.0f, 38.0f);
	continueButton->SetNumberOfFrames(1);
	continueButton->SetPosition(390.0f, 400.0f);
	continueButton->AddTexture("resources/continueButton.png");
	continueButton->IsVisible(true);
	continueButton->IsActive(false);
	inputManager->AddUiElement(continueButton);

	gameOverScreen = new Sprite(1);
	gameOverScreen->SetFrameSize(800.0f, 600.0f);
	gameOverScreen->SetNumberOfFrames(1);
	gameOverScreen->AddTexture("resources/gameover.png", false);
	gameOverScreen->IsActive(true);
	gameOverScreen->IsVisible(true);

	replayButton = new Sprite(1);
	replayButton->SetFrameSize(75.0f, 38.0f);
	replayButton->SetNumberOfFrames(1);
	replayButton->SetPosition(390.0f, 400.0f);
	replayButton->AddTexture("resources/replayButton.png");
	replayButton->IsVisible(true);
	replayButton->IsActive(false);
	inputManager->AddUiElement(replayButton);

	return true;
}

void CheckCollisions()
{
	if (player->IntersectsCircle(pickup))
	{
		pickup->IsVisible(false);
		pickup->IsActive(false);
		player->SetValue(player->GetValue() + pickup->GetValue());
		pickupSpawnTimer = 0.0f;
		pickupsReceived++;
	}
	if (player->IntersectsRect(enemy))
	{
		enemy->IsVisible(false);
		enemy->IsActive(false);
		player->SetValue(player->GetValue() + enemy->GetValue());
		enemySpawnTimer = 0.0f;
		enemiesHit++;
	}
}

void SpawnEnemy(float p_DeltaTime)
{
	if (enemy->IsVisible() == false)
	{
		enemySpawnTimer += p_DeltaTime;
		if (enemySpawnTimer >enemySpawnThreshold)
		{
			float marginX = enemy->GetSize().width;
			float marginY = enemy->GetSize().height;
			float spawnX = (rand() % (int)(screen_width - (marginX * 2))) + marginX;
			float spawnY = screen_height - ((rand() % (int)(player ->GetSize().height - (marginY * 2))) + marginY);
			enemy->SetPosition(spawnX, spawnY);
			enemy->IsVisible(true);
			enemy->IsActive(true);
		}
	}
}

void SpawnPickup(float p_DeltaTime)
{
	if (pickup->IsVisible() == false)
	{
		pickupSpawnTimer += p_DeltaTime;
		if (pickupSpawnTimer > pickupSpawnThreshold)
		{
			float marginX = pickup->GetSize().width;
			float marginY = pickup->GetSize().height;
			float spawnX = (rand() % (int)(screen_width - (marginX * 2))) + marginX;
			float spawnY = screen_height - ((rand() % (int)(player->GetSize().height - (marginY * 1.5))) + marginY );
			pickup->SetPosition(spawnX, spawnY);
			pickup->IsVisible(true);
			pickup->IsActive(true);
			pickupSpawnTimer = 0.0f;
		}
	}
}

void CheckBackground()
{
	float leftThreshold = 0.0f;
	float rightThreshold = -(background->GetSize().width - screen_width);
	if (background->GetPosition().x > 0)
	{
		background->SetPosition(0.0f, background->GetPosition().y);
	}
	else if (background->GetPosition().x < rightThreshold)
	{
		background->SetPosition(rightThreshold, background ->GetPosition().y);
	}
}

void CheckBoundaries(Sprite* p_sprite)
{
	Sprite::Rect check = p_sprite->GetCollisionRect();
	float offset;
	float x;
	float y;
	if (check.left < 0.0f)
	{
		p_sprite->SetVelocity(0.0f);
		offset = check.left;
		x = p_sprite->GetPosition().x - offset;
		y = p_sprite->GetPosition().y;
		p_sprite->SetPosition(x, y);
	}
	else if (check.right > screen_width)
	{
		p_sprite->SetVelocity(0.0f);
		offset = screen_width - check.right;
		x = p_sprite->GetPosition().x + offset;
		y = p_sprite->GetPosition().y;
		p_sprite->SetPosition(x, y);
	}
	if (check.top < 0.0f)
	{
		p_sprite->SetVelocity(0.0f);
		offset = check.top;
		y = p_sprite->GetPosition().y - offset;
		x = p_sprite->GetPosition().x;
		p_sprite->SetPosition(x, y);
	}
	else if (check.bottom > screen_height)
	{
		p_sprite->SetVelocity(0.0f);
		offset = screen_height - check.bottom;
		y = p_sprite->GetPosition().y + offset;
		x = p_sprite->GetPosition().x;
		p_sprite->SetPosition(x, y);
	}
}
void RestartGame()
{
	player->SetValue(0);
	robot_right->SetValue(0);
	robot_left->SetValue(0);
	pickupSpawnThreshold = 5.0f;
	pickupSpawnTimer = 0.0f;
	enemySpawnThreshold = 7.0f;
	enemySpawnTimer = 0.0f;
	splashDisplayTimer = 0.0f;
	splashDisplayThreshold = 5.0f;
	levelTimer = 0.0f;
	pickupsReceived = 0;
	pickupsThreshold = 5;
	enemiesHit = 0;
	pickup->IsVisible(false);
	enemy->IsVisible(false);
	background->SetVelocity(0.0f);
	robot_left->SetPosition(screen_width / 2.0f - 50.0f, screen_height - 130.0f);
	robot_left->IsVisible(false);
	robot_right->SetPosition(screen_width / 2.0f - 50.0f, screen_height - 130.0f);
	player = robot_right;
	player->IsActive(true);
	player->IsVisible(true);
	player->SetVelocity(0.0f);
}

void NextLevel()
{
	if (pickupsReceived < pickupsThreshold)
	{
		m_gameState = GameState::GS_GameOver;
	}
	else
	{
		pickupSpawnThreshold += pickupSpawnAdjustment;
		levelTimer = 0.0f;
		m_gameState = GameState::GS_NextLevel;
	}
}

void ProcessInput(const float p_deltaTime)
{
	Input::Command command = inputManager->GetCommand();
	switch (m_gameState)
	{
	case GameState::GS_Splash:
	case GameState::GS_Loading:
	{
		return;
	}
	break;
	case GameState::GS_Menu:
	case GameState::GS_Credits:
	case GameState::GS_Paused:
	case GameState::GS_NextLevel:
	case GameState::GS_GameOver:
	{
		command = Input::Command::CM_UI;
	}
	break;
	case GameState::GS_Running:
	{
	}
		break;
	}

	uiTimer += p_deltaTime;
	if (uiTimer > UI_THRESHOLD)
	{
		uiTimer = 0.0f;
		switch (command)
		{
		case Input::Command::CM_STOP:
		{
			player->SetVelocity(0.0f);
			background->SetVelocity(0.0f);
		}
		break;

		case Input::Command::CM_LEFT:
		{
			if (player == robot_right)
			{
				robot_right->IsActive(false);
				robot_right->IsVisible(false);
				robot_left->SetPosition(robot_right->GetPosition());
				robot_left->SetValue(robot_right->GetValue());
			}
			player = robot_left;
			player->IsActive(true);
			player->IsVisible(true);
			player->SetVelocity(-50.0f);
			background->SetVelocity(50.0f);
		}
		break;

		case Input::Command::CM_RIGHT:
		{
			if (player == robot_left)
			{
				robot_left->IsActive(false);
				robot_left->IsVisible(false);
				robot_right->SetPosition(robot_left->GetPosition());
				robot_right->SetValue(robot_left->GetValue());
			}

			player = robot_right;
			player->IsActive(true);
			player->IsVisible(true);
			player->SetVelocity(50.0f);
			background->SetVelocity(-50.0f);
		}
		break;

		case Input::Command::CM_UP:
		{
			player->Jump(Sprite::SpriteState::UP);
		}
		break;

		case Input::Command::CM_DOWN:
		{
			player->Jump(Sprite::SpriteState::DOWN);
		}
		break;

		case Input::Command::CM_QUIT:
		{
			PostQuitMessage(0);
		}
		break;

		case Input::Command::CM_UI:
		{
			if (pauseButton->IsClicked())
			{
				pauseButton->IsClicked(false);
				pauseButton->IsVisible(false);
				pauseButton->IsActive(false);

				resumeButton->IsVisible(true);
				resumeButton->IsActive(true);
				m_gameState = GS_Paused;
			}

			if (resumeButton->IsClicked())
			{
				resumeButton->IsClicked(false);
				resumeButton->IsVisible(false);
				resumeButton->IsActive(false);

				pauseButton->IsVisible(true);
				pauseButton->IsActive(true);
				m_gameState = GS_Running;
			}
			if (playButton->IsClicked())
			{
				playButton->IsClicked(false);
				exitButton->IsActive(false);
				playButton->IsActive(false);
				creditsButton->IsActive(false);
				m_gameState = GameState::GS_Running;
			}
			if (creditsButton->IsClicked())
			{
				creditsButton->IsClicked(false);
				exitButton->IsActive(false);
				playButton->IsActive(false);
				creditsButton->IsActive(false);
				m_gameState = GameState::GS_Credits;
			}
			if (exitButton->IsClicked())
			{
				playButton->IsClicked(false);
				exitButton->IsActive(false);
				playButton->IsActive(false);
				creditsButton->IsActive(false);
				PostQuitMessage(0);
			}
			if (menuButton->IsClicked())
			{
				menuButton->IsClicked(false);
				menuButton->IsActive(false);
				m_gameState = GameState::GS_Menu;
			}
			if (continueButton->IsClicked())
			{
				pickupsReceived = 0;
				enemiesHit = 0;
				continueButton->IsClicked(false);
				continueButton->IsActive(false);
				m_gameState = GameState::GS_Running;
			}

			if (replayButton->IsClicked())
			{
				replayButton->IsClicked(false);
				replayButton->IsActive(false);
				exitButton->IsActive(false);
				RestartGame();
				m_gameState = GameState::GS_Running;
			}
		}
		break;
		}
	}
	command = Input::Command::CM_INVALID;
}

void Update(const float p_deltaTime)
{
	switch (m_gameState)
	{
	case GameState::GS_Splash:
	case GameState::GS_Loading:
	{
		splashScreen->Update(p_deltaTime);
		splashDisplayTimer += p_deltaTime;
		if (splashDisplayTimer > splashDisplayThreshold)
		{
			m_gameState = GameState::GS_Menu;
		}
	}
	break;
	case GameState::GS_Menu:
	{
		menuScreen->Update(p_deltaTime);
		playButton->IsActive(true);
		creditsButton->IsActive(true);
		exitButton->IsActive(true);
		playButton->Update(p_deltaTime);
		creditsButton->Update(p_deltaTime);
		exitButton->Update(p_deltaTime);
		inputManager->Update(p_deltaTime);
		ProcessInput(p_deltaTime);
	}
	break;

	case GameState::GS_Credits:
	{
		creditsScreen->Update(p_deltaTime);
		menuButton->IsActive(true);
		menuButton->Update(p_deltaTime);
		inputManager->Update(p_deltaTime);
		ProcessInput(p_deltaTime);
	}
	break;
	case GameState::GS_Running:
	{
		inputManager->Update(p_deltaTime);
		ProcessInput(p_deltaTime);
		CheckBoundaries(player);
		CheckBackground();
		background->Update(p_deltaTime);
		robot_left->Update(p_deltaTime);
		robot_right->Update(p_deltaTime);
		robot_left_strip->Update(p_deltaTime);
		robot_right_strip->Update(p_deltaTime);
		pauseButton->Update(p_deltaTime);
		resumeButton->Update(p_deltaTime);
		pickup->Update(p_deltaTime);
		SpawnPickup(p_deltaTime);
		SpawnEnemy(p_deltaTime);
		enemy->Update(p_deltaTime);
		CheckCollisions();

		levelTimer += p_deltaTime;
		if (levelTimer > levelMaxTime)
		{
			NextLevel();
		}
	}
	break;
	case GameState::GS_Paused:
	{
		inputManager->Update(p_deltaTime);
		ProcessInput(p_deltaTime);
	}
	break;
	case GameState::GS_NextLevel:
	{
		nextLevelScreen->Update(p_deltaTime);
		continueButton->IsActive(true);
		continueButton->Update(p_deltaTime);
		inputManager->Update(p_deltaTime);
		ProcessInput(p_deltaTime);
	}
	break;
	case GameState::GS_GameOver:
	{
		gameOverScreen->Update(p_deltaTime);
		replayButton->IsActive(true);
		replayButton->Update(p_deltaTime);
		exitButton->IsActive(true);
		exitButton->Update(p_deltaTime);
		inputManager->Update(p_deltaTime);
		ProcessInput(p_deltaTime);
	}
	break;
	}
}

void Render()
{
	glClear(GL_COLOR_BUFFER_BIT);
	glLoadIdentity();

	switch (m_gameState)
	{
	case GameState::GS_Splash:
	case GameState::GS_Loading:
	{
		splashScreen->Render();
	}
	break;
	case GameState::GS_Menu:
	{
		menuScreen->Render();
		playButton->Render();
		creditsButton->Render();
		exitButton->Render();
	}
	break;
	case GameState::GS_Credits:
	{
		creditsScreen->Render();
		menuButton->Render();
		DrawCredits();
	}
	break;
	case GameState::GS_Running:
	case GameState::GS_Paused:
	{
		background->Render();
		robot_left->Render();
		robot_right->Render();
		robot_left_strip->Render();
		robot_right_strip->Render();
		pauseButton->Render();
		resumeButton->Render();
		pickup->Render();
		enemy->Render();
		DrawScore();
	}
	break;
	case GameState::GS_NextLevel:
	{
		nextLevelScreen->Render();
		DrawStats();
		continueButton->Render();
	}
	break;
	case GameState::GS_GameOver:
	{
		gameOverScreen->Render();
		replayButton->Render();
		DrawStats();
	}
	break;
	}

	SwapBuffers(hDC);
}
void StartGame()
{
	LoadSplash();
	inputManager = new Input(hWnd);

	uiTimer = 0.0f;
	srand(time(NULL));

	pickupSpawnThreshold = 3.0f;
	pickupSpawnTimer = 0.0f;

	enemySpawnThreshold = 7.0f;
	enemySpawnTimer = 0.0f;

	splashDisplayTimer = 0.0f;
	splashDisplayThreshold = 5.0f;

	levelTimer = 0.0f;
	levelMaxTime = 30.0f;
	pickupSpawnAdjustment = 0.25f;
	pickupsReceived = 0;
	pickupsThreshold = 5;
	enemiesHit = 0;

}

void GameLoop(const float p_deltatTime)
{
	if (m_gameState == GameState::GS_Splash)
	{
		BuildFont();
		LoadTextures();
		m_gameState = GameState::GS_Loading;
	}
	Update(p_deltatTime);
	Render();
}

void EndGame()
{
}

int APIENTRY wWinMain(_In_ HINSTANCE hInstance,
                     _In_opt_ HINSTANCE hPrevInstance,
                     _In_ LPWSTR    lpCmdLine,
                     _In_ int       nCmdShow)
{
    UNREFERENCED_PARAMETER(hPrevInstance);
    UNREFERENCED_PARAMETER(lpCmdLine);

    // TODO: Place code here.

    // Initialize global strings
    LoadStringW(hInstance, IDS_APP_TITLE, szTitle, MAX_LOADSTRING);
    LoadStringW(hInstance, IDC_ROBORACER2D, szWindowClass, MAX_LOADSTRING);
    MyRegisterClass(hInstance);

    // Perform application initialization:
    //if (!InitInstance (hInstance, nCmdShow))
    //{
    //    return FALSE;
    //}

	if (!CreateGLWindow("RoboRacer 2D", 800, 600, 16, false))
	{
		return false;
	}


    HACCEL hAccelTable = LoadAccelerators(hInstance, MAKEINTRESOURCE(IDC_ROBORACER2D));

    MSG msg;

	StartGame();
	//Game Loop
	int previousTime = glutGet(GLUT_ELAPSED_TIME);
	bool done = false;
	while (!done)
	{
		if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
		{
			if (msg.message == WM_QUIT)
			{
				done = true;
			}
			else
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}
		else
		{
			int currentTime = glutGet(GLUT_ELAPSED_TIME);
			float deltaTime = (float)(currentTime - previousTime) / 1000;
			previousTime = currentTime;
			GameLoop(deltaTime);
		}
	}
	EndGame();

    return (int) msg.wParam;
}



//
//  FUNCTION: MyRegisterClass()
//
//  PURPOSE: Registers the window class.
//
ATOM MyRegisterClass(HINSTANCE hInstance)
{
    WNDCLASSEXW wcex;

    wcex.cbSize = sizeof(WNDCLASSEX);

    wcex.style          = CS_HREDRAW | CS_VREDRAW;
    wcex.lpfnWndProc    = WndProc;
    wcex.cbClsExtra     = 0;
    wcex.cbWndExtra     = 0;
    wcex.hInstance      = hInstance;
    wcex.hIcon          = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ROBORACER2D));
    wcex.hCursor        = LoadCursor(nullptr, IDC_ARROW);
    wcex.hbrBackground  = (HBRUSH)(COLOR_WINDOW+1);
    wcex.lpszMenuName   = MAKEINTRESOURCEW(IDC_ROBORACER2D);
    wcex.lpszClassName  = szWindowClass;
    wcex.hIconSm        = LoadIcon(wcex.hInstance, MAKEINTRESOURCE(IDI_SMALL));

    return RegisterClassExW(&wcex);
}

//
//   FUNCTION: InitInstance(HINSTANCE, int)
//
//   PURPOSE: Saves instance handle and creates main window
//
//   COMMENTS:
//
//        In this function, we save the instance handle in a global variable and
//        create and display the main program window.
//
BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)
{
   hInst = hInstance; // Store instance handle in our global variable

   HWND hWnd = CreateWindowW(szWindowClass, szTitle, WS_OVERLAPPEDWINDOW,
      CW_USEDEFAULT, 0, CW_USEDEFAULT, 0, nullptr, nullptr, hInstance, nullptr);

   if (!hWnd)
   {
      return FALSE;
   }

   ShowWindow(hWnd, nCmdShow);
   UpdateWindow(hWnd);

   return TRUE;
}

//
//  FUNCTION: WndProc(HWND, UINT, WPARAM, LPARAM)
//
//  PURPOSE:  Processes messages for the main window.
//
//  WM_COMMAND  - process the application menu
//  WM_PAINT    - Paint the main window
//  WM_DESTROY  - post a quit message and return
//
//
LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message)
    {
    case WM_COMMAND:
        {
            int wmId = LOWORD(wParam);
            // Parse the menu selections:
            switch (wmId)
            {
            case IDM_ABOUT:
                DialogBox(hInst, MAKEINTRESOURCE(IDD_ABOUTBOX), hWnd, About);
                break;
            case IDM_EXIT:
                DestroyWindow(hWnd);
                break;
            default:
                return DefWindowProc(hWnd, message, wParam, lParam);
            }
        }
        break;
    case WM_PAINT:
        {
            PAINTSTRUCT ps;
            HDC hdc = BeginPaint(hWnd, &ps);
            // TODO: Add any drawing code that uses hdc here...
            EndPaint(hWnd, &ps);
        }
        break;
    case WM_DESTROY:
        PostQuitMessage(0);
        break;
    default:
        return DefWindowProc(hWnd, message, wParam, lParam);
    }
    return 0;
}

// Message handler for about box.
INT_PTR CALLBACK About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    UNREFERENCED_PARAMETER(lParam);
    switch (message)
    {
    case WM_INITDIALOG:
        return (INT_PTR)TRUE;

    case WM_COMMAND:
        if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL)
        {
            EndDialog(hDlg, LOWORD(wParam));
            return (INT_PTR)TRUE;
        }
        break;
    }
    return (INT_PTR)FALSE;
}

void Sprite::Jump(SpriteState p_state)
{
	if (p_state == SpriteState::DOWN)
	{
		if (m_position.y < 470.0f) m_position.y += 75.0f;
	}
	else if (p_state == SpriteState::UP)
	{
		if (m_position.y >= 470.0f) m_position.y -= 75.0f;
	}
}
